/****************************************************************************
*
* Copyright (c) 2015 Broadcom Corporation
*
* Unless you and Broadcom execute a separate written software license
* agreement governing use of this software, this software is licensed to
* you under the terms of the GNU General Public License version 2 (the
* "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
* the following added to such license:
*
* As a special exception, the copyright holders of this software give you
* permission to link this software with independent modules, and to copy
* and distribute the resulting executable under terms of your choice,
* provided that you also meet, for each linked independent module, the
* terms and conditions of the license of that module. An independent
* module is a module which is not derived from this software. The special
* exception does not apply to any modifications of the software.
*
* Notwithstanding the above, under no circumstances may you combine this
* software in any way with any other Broadcom software provided under a
* license other than the GPL, without Broadcom's express prior written
* consent.
*
****************************************************************************
* Author: Sandep Nanda <snanda@broadcom.com>
*****************************************************************************/
#include <linux/init.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/delay.h>
#include <linux/platform_device.h>
#include <linux/interrupt.h>
#include <linux/gpio/consumer.h>
#include <linux/stat.h>
#include <linux/string.h>
#include <linux/fs.h>
#include <linux/debugfs.h>
#include <linux/cdev.h>
#include <asm/signal.h>
#include <asm/siginfo.h>
#include <linux/version.h>
#include <linux/of.h>
#include <linux/of_irq.h>
#include <linux/io.h>
#include <linux/types.h>
#include <linux/slab.h>
#include <uapi/linux/bcm_media_gw/wps/wps_gpio_api.h>
#include <linux/uaccess.h>
#include <linux/irq.h>

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Sandep Nanda");
MODULE_DESCRIPTION("Broadcom CM WPS-GPIO Driver");
MODULE_VERSION("1.0");

static int node_count;

struct wps_gpio_priv {
	struct device *dev;
	struct platform_device *pdev;
	struct task_struct *task;
	dev_t ndev;
	struct cdev c_dev;
	struct class *wps_char_class;
	struct device *wps_char_device;
	struct fasync_struct *sigio_list;
	int irq;
	struct gpio_desc *gpio_pbc;
	struct gpio_desc *gpio_led[32];
	int num_led;
};

static int wps_gpio_open(struct inode *inode, struct file *flp)
{
	struct wps_gpio_priv *priv; /* device information */
	priv = container_of(inode->i_cdev, struct wps_gpio_priv, c_dev);
	flp->private_data = priv; /* for other methods */

	return 0;
}

static int wps_gpio_async(int fd, struct file *filp, int onflag)
{
	struct wps_gpio_priv *priv = filp->private_data;
	return fasync_helper(fd, filp, onflag, &priv->sigio_list);
}

static int wps_gpio_release(struct inode *inode, struct file *flp)
{
	wps_gpio_async(-1, flp, 0);

	return 0;
}

static int set_led_state(void *arg, struct wps_led_state *led_state)
{
	struct wps_gpio_priv *priv = arg;
	int led_count = 0;
	unsigned long inval_chk_mask = (unsigned long)(-1) << priv->num_led;

	if ((led_state->mask & inval_chk_mask) ||
	     (led_state->value & inval_chk_mask))
		return -EINVAL;

	for (led_count = 0; led_count < priv->num_led; led_count++) {
		if (led_state->mask & ((unsigned long)1 << led_count)) {
			gpiod_set_value(priv->gpio_led[led_count],
					(led_state->value &
					    (1 << led_count)) >> led_count);
		}
	}
	return 0;
}

long wps_gpio_ioctl(struct file *filep, unsigned int cmd, unsigned long arg)
{
	struct wps_gpio_priv *priv = filep->private_data;
	struct wps_led_state led_state;
	int status = -EINVAL;
	if (cmd == BCM_WPS_SET_LED) {
		status = copy_from_user(&led_state, (void *)arg,
					sizeof(led_state));
		if (status)
			goto done;
		status = set_led_state(priv, &led_state);
		if (status)
			goto done;
	}

done:
	return status;
}

static struct file_operations wps_gpio_file_operations = {
	.owner = THIS_MODULE,
	.open = wps_gpio_open,
	.release =  wps_gpio_release,
	.unlocked_ioctl = wps_gpio_ioctl,
	.fasync = wps_gpio_async,
};

static irqreturn_t wps_gpio_pbc_isr(int irq, void *d)
{
	/*triger the signal to the user space*/
	struct wps_gpio_priv *priv;
	printk("WPS IRQ triggered\n");
	priv = d;
	kill_fasync(&priv->sigio_list, SIGIO, POLL_IN);
	return IRQ_HANDLED;
}

static int wps_gpio_probe(struct platform_device *pdev)
{
	int status = 0;
	int led_count = 0;
	struct device *dev = &pdev->dev;
	struct wps_gpio_priv *priv;

	u32 irqflags;
	struct device_node *of_node = pdev->dev.of_node;
	const char *ptr = NULL;
	char device_name[10], led_name[10];

	dev_info(dev, "WPS LED/PBC: Initializing the WPS  LKM-->\n");
	node_count++;
	/*Do not allow probing of more than one node*/
	if (node_count > 1) {
		status = -EINVAL;
		goto done;
	}

	priv = devm_kzalloc(dev, sizeof(*priv), GFP_KERNEL);
	if (IS_ERR_OR_NULL(priv)) {
		dev_err(dev, "unable to allocate private data\n");
		status = -ENOMEM;
		goto done;
	}

	if (of_property_read_string(of_node, "dev-name", &ptr)) {
		/* Couldn't find the entry */
		snprintf(device_name, sizeof(device_name), "wps%d", node_count);
		device_name[sizeof(device_name) - 1] = '\0';
	} else {
		snprintf(device_name, sizeof(device_name), "%s", ptr);
		device_name[sizeof(device_name) - 1] = '\0';
	}

	priv->num_led = 0;

	for (led_count = 0; led_count < 32; led_count++) {
		snprintf(led_name, sizeof(led_name), "led%d", led_count);
		led_name[sizeof(led_name) - 1] = '\0';

		priv->gpio_led[led_count] =
			 devm_gpiod_get(&pdev->dev, led_name, GPIOD_OUT_LOW);

		if (IS_ERR(priv->gpio_led[led_count])) {
			if (PTR_ERR(priv->gpio_led[led_count]) == -ENOENT) {
				/*
				 * Its assumed that the LED's defined in the
				 * device tree are contigous
				 */
				break;
			} else {
				dev_err(dev, "Couldn't get GPIO for WPS %s ",
					 led_name);
				status = PTR_ERR(priv->gpio_led[led_count]);
				goto done;
			}
		}

		priv->num_led++;
	}

	priv->gpio_pbc = devm_gpiod_get(&pdev->dev, "pbc", GPIOD_IN);
	if (IS_ERR(priv->gpio_pbc)) {
		dev_err(dev, "Couldn't get GPIO for WPS PUSH-BUTTON");
		status = -ENOENT;
		goto done;
	}

	priv->irq = platform_get_irq(pdev, 0);
	if (priv->irq < 0) {
		dev_err(dev, "Couldn't get IRQ\n");
		status = priv->irq;
		goto done;
	}

	irqflags = irq_get_trigger_type(priv->irq);
	dev_err(dev, "Requesting IRQ %d with flags 0x%08x for GPIO %d.\n",
		priv->irq, irqflags, desc_to_gpio(priv->gpio_pbc));

	status = devm_request_irq(dev, priv->irq, wps_gpio_pbc_isr, irqflags,
				  "WPS IRQ", priv);
	if (status) {
		dev_err(dev, "Failed registering IRQ with status %d!\n",
			status);
		goto done;
	}

	platform_set_drvdata(pdev, priv);
	priv->pdev = pdev;

	status = alloc_chrdev_region(&priv->ndev, 0, 1, device_name);
	if (status) {
		dev_err(dev, "Failed registering char device number "
			     "with status %d!\n", status);
		goto done;
	}

	cdev_init(&priv->c_dev, &wps_gpio_file_operations);
	(&priv->c_dev)->owner = THIS_MODULE;
	cdev_add(&priv->c_dev, priv->ndev, node_count);

	priv->wps_char_class = class_create(THIS_MODULE, device_name);
	if (IS_ERR(priv->wps_char_class)) {
		cdev_del(&priv->c_dev);
		unregister_chrdev_region(priv->ndev, 1);
		status = PTR_ERR(priv->wps_char_class);
		goto done;
	}

	priv->wps_char_device = device_create(priv->wps_char_class,
					      NULL, priv->ndev, NULL,
					      device_name);
	if (IS_ERR(priv->wps_char_device)) {
		class_destroy(priv->wps_char_class);
		cdev_del(&priv->c_dev);
		unregister_chrdev_region(priv->ndev, 1);
		status = PTR_ERR(priv->wps_char_device);
		goto done;
	}

done:
	dev_dbg(dev, "<--\n");
	return status;
}

static int wps_gpio_remove(struct platform_device *pdev)
{
	int led_count, status = 0;
	struct device *dev = &pdev->dev;
	struct wps_gpio_priv *priv = platform_get_drvdata(pdev);

	if (!priv) {
		dev_err(dev, "Release called with uninitialized\n");
		status = -EINVAL;
		goto done;
	}

	disable_irq(priv->irq);
	for (led_count = 0; led_count < priv->num_led; led_count++)
		devm_gpiod_put(dev, priv->gpio_led[led_count]);

	devm_gpiod_put(dev, priv->gpio_pbc);
	class_destroy(priv->wps_char_class);
	cdev_del(&priv->c_dev);
	unregister_chrdev_region(priv->ndev, 1);

done:
	return status;
}

static const struct of_device_id wps_gpio_of_match[] = {
	{ .compatible = "brcm,wps-gpio" },
	{ }
};

MODULE_DEVICE_TABLE(of, wps_gpio_of_match);

static struct platform_driver wps_gpio_driver = {
	.probe	= wps_gpio_probe,
	.remove	= wps_gpio_remove,
	.driver = {
		.name		= "wps_gpio",
		.owner		= THIS_MODULE,
		.of_match_table	= wps_gpio_of_match
	}
};

__init int wps_gpio_init(void)
{
	return platform_driver_register(&wps_gpio_driver);
}

void wps_gpio_exit(void)
{
	platform_driver_unregister(&wps_gpio_driver);
}

module_init(wps_gpio_init);
module_exit(wps_gpio_exit);
